-- Tinder Style Dating App Database Schema (Improved)

-- Create database if it doesn't exist (optional, depends on setup)
-- CREATE DATABASE IF NOT EXISTS dating_app CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
-- USE dating_app;

-- Membership Types Table
CREATE TABLE memberships (
    membership_id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL UNIQUE, -- e.g., Standard, Premium, Gold
    description TEXT
);

-- Users Table
CREATE TABLE users (
    user_id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    membership_id INT DEFAULT 1, -- Default to Standard membership
    is_admin BOOLEAN DEFAULT FALSE, -- Flag for admin users
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    online_status BOOLEAN DEFAULT FALSE, -- Track online/offline status
    INDEX idx_email (email), -- Index for login lookup
    FOREIGN KEY (membership_id) REFERENCES memberships(membership_id)
);

-- Profiles Table
CREATE TABLE profiles (
    profile_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL UNIQUE,
    username VARCHAR(100) NOT NULL,
    birthdate DATE NULL, -- Changed from age to birthdate for dynamic calculation
    location VARCHAR(255),
    interests TEXT, -- Could be comma-separated or JSON
    bio TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id), -- Index for joining with users
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
);

-- Photos Table
CREATE TABLE photos (
    photo_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    file_path VARCHAR(255) NOT NULL, -- Path to the uploaded photo
    upload_order INT DEFAULT 0, -- To maintain order, 0 could be profile pic
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id), -- Index for retrieving user photos
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
);

-- Likes/Dislikes Table
CREATE TABLE likes (
    like_id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL, -- The user performing the action
    liked_user_id INT NOT NULL, -- The user being liked/disliked
    action ENUM('like', 'dislike') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY user_action_target (user_id, liked_user_id), -- Prevent duplicate actions
    INDEX idx_user_id (user_id), -- Index for finding user's actions
    INDEX idx_liked_user_id (liked_user_id), -- Index for finding who liked a user
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (liked_user_id) REFERENCES users(user_id) ON DELETE CASCADE
);

-- Matches Table
CREATE TABLE matches (
    match_id INT AUTO_INCREMENT PRIMARY KEY,
    user1_id INT NOT NULL,
    user2_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_match (user1_id, user2_id), -- Ensure unique matches regardless of order
    INDEX idx_user1_id (user1_id), -- Index for finding user's matches
    INDEX idx_user2_id (user2_id), -- Index for finding user's matches
    FOREIGN KEY (user1_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (user2_id) REFERENCES users(user_id) ON DELETE CASCADE,
    CHECK (user1_id < user2_id) -- Ensure user1_id is always less than user2_id to prevent duplicates like (1,2) and (2,1)
);

-- Messages Table
CREATE TABLE messages (
    message_id INT AUTO_INCREMENT PRIMARY KEY,
    match_id INT NOT NULL,
    sender_id INT NOT NULL,
    receiver_id INT NOT NULL,
    content TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_read BOOLEAN DEFAULT FALSE,
    INDEX idx_match_id (match_id), -- Index for retrieving conversation messages
    INDEX idx_sender_id (sender_id),
    INDEX idx_receiver_id (receiver_id),
    INDEX idx_created_at (created_at), -- Index for ordering messages
    FOREIGN KEY (match_id) REFERENCES matches(match_id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (receiver_id) REFERENCES users(user_id) ON DELETE CASCADE
);

-- Insert initial membership types
INSERT INTO memberships (name, description) VALUES
('Standard', 'Basic free membership'),
('Premium', 'Enhanced features membership'),
('Gold', 'Top-tier membership with all features');

-- Note: Admin user and test data are in a separate file (test_data.sql)


