<?php
// Check if user is logged in
require_once __DIR__ . '/../includes/auth_check.php'; // Fixed path

// Include database configuration
require_once __DIR__ . '/../config/db.php'; // Fixed path

$mysqli = connect_db(); // Establish connection
$user_id = $_SESSION['user_id'];

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $target_user_id = filter_input(INPUT_POST, 'target_user_id', FILTER_VALIDATE_INT);
    $action = $_POST['action'] ?? ''; // 'like' or 'dislike'

    // Validate input
    if (!$target_user_id || ($action !== 'like' && $action !== 'dislike')) {
        // Redirect back with error - invalid request
        header("Location: ../public/find_matches.php?error=invalid_request");
        exit;
    }

    if ($target_user_id == $user_id) {
        // Redirect back with error - cannot like self
        header("Location: ../public/find_matches.php?error=self_action");
        exit;
    }

    // --- Database Interaction ---
    $match_found = false;

    // 1. Record the action (Like or Dislike)
    // Use INSERT ... ON DUPLICATE KEY UPDATE to handle existing likes/dislikes
    $sql_insert_like = "INSERT INTO likes (user_id, liked_user_id, action, created_at) VALUES (?, ?, ?, NOW()) 
                        ON DUPLICATE KEY UPDATE action = VALUES(action), created_at = NOW()";
    if ($stmt_insert = $mysqli->prepare($sql_insert_like)) {
        $stmt_insert->bind_param("iis", $user_id, $target_user_id, $action);
        if (!$stmt_insert->execute()) {
            error_log("Database error (record action): " . $stmt_insert->error);
            header("Location: ../public/find_matches.php?error=db_error");
            close_db($mysqli);
            exit;
        }
        $stmt_insert->close();
    } else {
        error_log("Database error (prepare record action): " . $mysqli->error);
        header("Location: ../public/find_matches.php?error=db_error");
        close_db($mysqli);
        exit;
    }

    // 2. If it was a 'like', check for a mutual like (match)
    if ($action === 'like') {
        $sql_check_match = "SELECT like_id FROM likes WHERE user_id = ? AND liked_user_id = ? AND action = 'like'";
        if ($stmt_check = $mysqli->prepare($sql_check_match)) {
            $stmt_check->bind_param("ii", $target_user_id, $user_id);
            $stmt_check->execute();
            $stmt_check->store_result();

            if ($stmt_check->num_rows > 0) {
                // It's a match!
                $match_found = true;

                // 3. Create match record (ensure user1_id < user2_id to prevent duplicates)
                $user1 = min($user_id, $target_user_id);
                $user2 = max($user_id, $target_user_id);
                $sql_insert_match = "INSERT IGNORE INTO matches (user1_id, user2_id, created_at) VALUES (?, ?, NOW())";
                if ($stmt_match = $mysqli->prepare($sql_insert_match)) {
                    $stmt_match->bind_param("ii", $user1, $user2);
                    if (!$stmt_match->execute()) {
                        // Log error, but don't stop the user flow
                        error_log("Database error (insert match): " . $stmt_match->error);
                    }
                    $stmt_match->close();
                } else {
                     error_log("Database error (prepare insert match): " . $mysqli->error);
                }
            }
            $stmt_check->close();
        } else {
             error_log("Database error (check match): " . $mysqli->error);
             // Continue, but log the error
        }
    }
    close_db($mysqli); // Close connection


    // --- Redirect ---
    // Redirect back to find matches page.
    // Pass a parameter if a match occurred.
    $redirect_url = "../public/find_matches.php";
    if ($match_found) {
        // Store match notification in session flash message for better UX
        $_SESSION['flash_message'] = "Tebrikler! Bir kullanıcı ile eşleştin!"; // Fetch username later if needed
        $redirect_url .= "?match=true"; // Simple flag for now
    }

    header("Location: " . $redirect_url);
    exit;

} else {
    // If not a POST request, redirect
    header("Location: ../public/find_matches.php");
    exit;
}
?>

