<?php
// Check if user is logged in
require_once __DIR__ . 
'/../includes/auth_check.php'; // Fixed path

// Include database configuration
require_once __DIR__ . 
'/../config/db.php'; // Fixed path

$mysqli = connect_db(); // Establish connection
$user_id = $_SESSION['user_id'];
$upload_dir = __DIR__ . '/../public/uploads/'; // Define upload directory relative to this script
$max_photos = 9;
$allowed_types = ['image/jpeg', 'image/png', 'image/webp', 'image/gif']; // Added GIF
$max_file_size = 5 * 1024 * 1024; // 5 MB

// Ensure upload directory exists and is writable
if (!is_dir($upload_dir)) {
    if (!mkdir($upload_dir, 0775, true)) {
        error_log("Failed to create upload directory: " . $upload_dir);
        header("Location: ../public/profile.php?error=upload_dir_creation");
        close_db($mysqli);
        exit;
    }
} elseif (!is_writable($upload_dir)) {
    error_log("Upload directory not writable: " . $upload_dir);
    // Attempt to make it writable (might fail depending on permissions)
    @chmod($upload_dir, 0775);
    if (!is_writable($upload_dir)) {
        header("Location: ../public/profile.php?error=upload_dir_permission");
        close_db($mysqli);
        exit;
    }
}


if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // --- Process Profile Information --- 
    $username = trim($_POST['username'] ?? '');
    $age = filter_input(INPUT_POST, 'age', FILTER_VALIDATE_INT, ['options' => ['min_range' => 18]]);
    $location = trim($_POST['location'] ?? '');
    $interests = trim($_POST['interests'] ?? '');
    $bio = trim($_POST['bio'] ?? '');

    // Basic validation
    if (empty($username)) {
        header("Location: ../public/profile.php?error=username_required");
        close_db($mysqli);
        exit;
    }

    // --- Database Update for Profile ---
    
    // Check if profile exists, if not, INSERT, otherwise UPDATE
    $sql_check = "SELECT profile_id FROM profiles WHERE user_id = ?";
    if ($stmt_check = $mysqli->prepare($sql_check)) {
        $stmt_check->bind_param("i", $user_id);
        $stmt_check->execute();
        $stmt_check->store_result();
        $profile_exists = $stmt_check->num_rows > 0;
        $stmt_check->close();

        if ($profile_exists) {
            $sql_update_profile = "UPDATE profiles SET username = ?, age = ?, location = ?, interests = ?, bio = ? WHERE user_id = ?";
            if ($stmt_update = $mysqli->prepare($sql_update_profile)) {
                // Use null for age if it's not provided or invalid
                $age_to_save = ($age === false || $age === null) ? null : $age;
                $stmt_update->bind_param("sisssi", $username, $age_to_save, $location, $interests, $bio, $user_id);
                if (!$stmt_update->execute()) {
                     error_log('Profile update failed: ' . $stmt_update->error);
                     header("Location: ../public/profile.php?error=profile_update_failed");
                     close_db($mysqli);
                     exit;
                }
                $stmt_update->close();
            }
        } else {
            $sql_insert_profile = "INSERT INTO profiles (user_id, username, age, location, interests, bio) VALUES (?, ?, ?, ?, ?, ?)";
            if ($stmt_insert = $mysqli->prepare($sql_insert_profile)) {
                 // Use null for age if it's not provided or invalid
                $age_to_save = ($age === false || $age === null) ? null : $age;
                $stmt_insert->bind_param("isisss", $user_id, $username, $age_to_save, $location, $interests, $bio);
                if (!$stmt_insert->execute()) {
                    error_log('Profile creation failed: ' . $stmt_insert->error);
                    header("Location: ../public/profile.php?error=profile_create_failed");
                    close_db($mysqli);
                    exit;
                }
                $stmt_insert->close();
            }
        }
    } else {
         error_log('Database error (profile check): ' . $mysqli->error);
         header("Location: ../public/profile.php?error=db_error");
         close_db($mysqli);
         exit;
    }
    

    // --- Process Photo Uploads --- 
    $upload_errors = [];
    $uploaded_files_count = 0;

    // Count existing photos
    $existing_photos_count = 0;
    $sql_count = "SELECT COUNT(*) as count FROM photos WHERE user_id = ?";
    if ($stmt_count = $mysqli->prepare($sql_count)) {
        $stmt_count->bind_param("i", $user_id);
        $stmt_count->execute();
        $result_count = $stmt_count->get_result();
        $row_count = $result_count->fetch_assoc();
        $existing_photos_count = $row_count['count'];
        $stmt_count->close();
    } else {
        error_log('Database error (photo count): ' . $mysqli->error);
        // Don't stop the whole process, but log the error
    }

    if (isset($_FILES['photos']) && !empty($_FILES['photos']['name'][0])) {
        $total_files = count($_FILES['photos']['name']);
        $allowed_uploads = $max_photos - $existing_photos_count;

        for ($i = 0; $i < $total_files && $uploaded_files_count < $allowed_uploads; $i++) {
            $file_name = $_FILES['photos']['name'][$i];
            $file_tmp = $_FILES['photos']['tmp_name'][$i];
            $file_size = $_FILES['photos']['size'][$i];
            $file_error = $_FILES['photos']['error'][$i];
            $file_type = mime_content_type($file_tmp); // More reliable type check

            if ($file_error === UPLOAD_ERR_OK) {
                // Validate type
                if (!in_array($file_type, $allowed_types)) {
                    $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Geçersiz dosya türü ($file_type).";
                    continue;
                }
                // Validate size
                if ($file_size > $max_file_size) {
                    $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Dosya boyutu çok büyük (Maks: " . ($max_file_size / 1024 / 1024) . " MB).";
                    continue;
                }

                // Generate unique filename
                $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                $new_file_name = uniqid('user' . $user_id . '_', true) . '.' . $file_ext;
                $destination = $upload_dir . $new_file_name;
                $db_file_path = 'uploads/' . $new_file_name; // Path to store in DB (relative to public root)

                // Move the file
                if (move_uploaded_file($file_tmp, $destination)) {
                    $uploaded_files_count++;
                    // --- Database Insert for Photo ---
                    
                    $upload_order = $existing_photos_count + $uploaded_files_count; // Simple order
                    $sql_insert_photo = "INSERT INTO photos (user_id, file_path, upload_order) VALUES (?, ?, ?)";
                    if ($stmt_photo = $mysqli->prepare($sql_insert_photo)) {
                        $stmt_photo->bind_param("isi", $user_id, $db_file_path, $upload_order);
                        if (!$stmt_photo->execute()) {
                            $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Veritabanına kaydedilemedi: " . $stmt_photo->error;
                            // Optionally delete the uploaded file if DB insert fails
                            unlink($destination);
                            $uploaded_files_count--; 
                        }
                        $stmt_photo->close();
                    } else {
                         $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Veritabanı hatası (fotoğraf ekleme).";
                         unlink($destination);
                         $uploaded_files_count--; 
                    }
                    
                } else {
                    $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Dosya yüklenemedi (sunucu hatası).";
                }
            } elseif ($file_error !== UPLOAD_ERR_NO_FILE) {
                $upload_errors[] = "'" . htmlspecialchars($file_name) . "': Yükleme hatası (kod: $file_error).";
            }
        }
    }

    // --- Redirect or show messages ---
    close_db($mysqli);

    $status_message = "updated";
    $error_query_string = "";
    if (!empty($upload_errors)) {
        // Store errors in session for better display on the profile page
        $_SESSION['upload_errors'] = $upload_errors;
        $status_message = "updated_with_errors";
    }

    header("Location: ../public/profile.php?status=" . $status_message);
    exit;

} else {
    // If not a POST request, redirect to profile page
    header("Location: ../public/profile.php");
    exit;
}
?>

