<?php
session_start(); // Start the session at the beginning

require_once __DIR__ . '/db.php'; // Path assumes db.php is in the same directory as this script

// Establish database connection
$mysqli = connect_db();

// Check if the form was submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];

    // --- Basic Validation ---
    if (empty($email) || empty($password) || empty($confirm_password)) {
        close_db($mysqli);
        header("Location: ../public/register.php?error=empty_fields");
        exit;
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        close_db($mysqli);
        header("Location: ../public/register.php?error=invalid_email");
        exit;
    }

    if ($password !== $confirm_password) {
        close_db($mysqli);
        header("Location: ../public/register.php?error=password_mismatch");
        exit;
    }

    // --- Password Hashing ---
    $password_hash = password_hash($password, PASSWORD_DEFAULT);

    // --- Database Interaction ---

    // Check if email already exists
    $sql_check = "SELECT user_id FROM users WHERE email = ?";
    if ($stmt_check = $mysqli->prepare($sql_check)) {
        $stmt_check->bind_param("s", $email);
        $stmt_check->execute();
        $stmt_check->store_result();

        if ($stmt_check->num_rows > 0) {
            $stmt_check->close();
            close_db($mysqli);
            header("Location: ../public/register.php?error=email_exists");
            exit;
        }
        $stmt_check->close();
    } else {
        error_log("Database error (check email): " . $mysqli->error);
        close_db($mysqli);
        header("Location: ../public/register.php?error=db_error");
        exit;
    }

    // Insert new user
    $default_membership_id = 1;
    $sql_insert = "INSERT INTO users (email, password_hash, membership_id, created_at, is_admin) VALUES (?, ?, ?, NOW(), 0)"; // Added is_admin default
    if ($stmt_insert = $mysqli->prepare($sql_insert)) {
        $stmt_insert->bind_param("ssi", $email, $password_hash, $default_membership_id);

        if ($stmt_insert->execute()) {
            $new_user_id = $mysqli->insert_id;
            $stmt_insert->close();

            // Automatically create a basic profile entry
            $default_username = 'User' . $new_user_id; // Or generate from email part
            $sql_profile = "INSERT INTO profiles (user_id, username) VALUES (?, ?)";
            if ($stmt_profile = $mysqli->prepare($sql_profile)) {
                $stmt_profile->bind_param("is", $new_user_id, $default_username);
                $stmt_profile->execute();
                $stmt_profile->close();
            } else {
                 error_log("Database error (create profile): " . $mysqli->error);
                 // Continue registration even if profile creation fails initially
            }

            close_db($mysqli);
            header("Location: ../public/login.php?registration=success");
            exit;
        } else {
            error_log("Database error (insert user): " . $stmt_insert->error);
            $stmt_insert->close();
            close_db($mysqli);
            header("Location: ../public/register.php?error=registration_failed");
            exit;
        }
    } else {
        error_log("Database error (prepare insert user): " . $mysqli->error);
        close_db($mysqli);
        header("Location: ../public/register.php?error=db_error");
        exit;
    }

} else {
    // If not a POST request, redirect to registration page
    close_db($mysqli); // Close connection if script accessed directly
    header("Location: ../public/register.php");
    exit;
}
?>