<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
require_once __DIR__ . '/../includes/auth_check.php'; // Corrected path

// Include database configuration
require_once __DIR__ . '/../config/db.php'; // Corrected path

$user_id = $_SESSION['user_id'];
$matches_data = [];

// Establish database connection
$mysqli = connect_db();

// Fetch matches and last message for each match
$sql_matches = "
    SELECT 
        m.match_id, 
        CASE 
            WHEN m.user1_id = ? THEN m.user2_id 
            ELSE m.user1_id 
        END AS other_user_id,
        p.username AS other_username,
        (SELECT file_path FROM photos ph WHERE ph.user_id = other_user_id ORDER BY ph.upload_order ASC LIMIT 1) AS profile_pic,
        (SELECT content FROM messages msg WHERE msg.match_id = m.match_id ORDER BY msg.created_at DESC LIMIT 1) AS last_message_preview,
        (SELECT created_at FROM messages msg WHERE msg.match_id = m.match_id ORDER BY msg.created_at DESC LIMIT 1) AS last_message_time
    FROM matches m
    JOIN profiles p ON p.user_id = CASE WHEN m.user1_id = ? THEN m.user2_id ELSE m.user1_id END
    WHERE m.user1_id = ? OR m.user2_id = ?
    ORDER BY last_message_time DESC, m.created_at DESC; -- Order by last message time, then by match time
";

if ($stmt_matches = $mysqli->prepare($sql_matches)) {
    $stmt_matches->bind_param("iiii", $user_id, $user_id, $user_id, $user_id);
    $stmt_matches->execute();
    $result_matches = $stmt_matches->get_result();
    while ($row = $result_matches->fetch_assoc()) {
        // Format last_message_time nicely (e.g., '10:30', 'Dün', '01.05.2025')
        if ($row['last_message_time']) {
            $now = new DateTime();
            $last_msg_time = new DateTime($row['last_message_time']);
            $interval = $now->diff($last_msg_time);
            if ($interval->d == 0) {
                $row['last_message_time_formatted'] = $last_msg_time->format('H:i'); // Today
            } elseif ($interval->d == 1) {
                $row['last_message_time_formatted'] = 'Dün'; // Yesterday
            } else {
                $row['last_message_time_formatted'] = $last_msg_time->format('d.m.Y'); // Older date
            }
        } else {
            $row['last_message_time_formatted'] = ''; // No messages yet
        }
        
        // Adjust profile picture path
        $row['profile_pic'] = $row['profile_pic'] ? '../' . $row['profile_pic'] : '../public/images/default_avatar.png'; // Default avatar if no photo

        $matches_data[] = $row;
    }
    $stmt_matches->close();
} else {
    error_log("Messages DB Error (fetch matches): " . $mysqli->error);
    // Handle error appropriately, maybe display a message to the user
}

close_db($mysqli);

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mesajlar - Infernova</title>
    <link rel="stylesheet" href="../css/style.css"> <!-- Assuming a common CSS file -->
    <style>
        /* Specific styles for messages page */
        .container {
            max-width: 800px;
            margin: 30px auto;
            padding: 20px;
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .match-list ul {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        .match-list li {
            display: flex;
            align-items: center;
            padding: 15px;
            border-bottom: 1px solid #eee;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }
        .match-list li:last-child {
            border-bottom: none;
        }
        .match-list li:hover {
            background-color: #f9f9f9;
        }
        .match-list img {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            margin-right: 15px;
            object-fit: cover;
        }
        .match-info {
            flex-grow: 1;
            overflow: hidden; /* Prevent long text from breaking layout */
        }
        .match-info h4 {
            margin: 0 0 5px;
            font-size: 1.1em;
            color: #333;
        }
        .match-info p {
            margin: 0;
            font-size: 0.9em;
            color: #555;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .time {
            font-size: 0.8em;
            color: #999;
            margin-left: 15px;
            white-space: nowrap;
        }
        .no-matches {
            text-align: center;
            color: #777;
            padding: 40px 20px;
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="logo">
            <a href="dashboard.php">Infernova</a>
        </div>
        <div class="nav-links">
            <a href="profile.php">Profilim</a>
            <a href="find_matches.php">Eşleşme Bul</a>
            <a href="messages.php" class="active">Mesajlar</a>
            <a href="../includes/logout.php">Çıkış Yap</a>
        </div>
    </nav>

    <div class="container">
        <h2>Mesajlar</h2>

        <div class="match-list">
            <?php if (!empty($matches_data)): ?>
                <ul>
                    <?php foreach ($matches_data as $match): ?>
                        <li onclick="window.location.href='conversation.php?match_id=<?php echo $match['match_id']; ?>&user_id=<?php echo $match['other_user_id']; ?>'">
                            <img src="<?php echo htmlspecialchars($match['profile_pic']); ?>" alt="<?php echo htmlspecialchars($match['other_username']); ?>">
                            <div class="match-info">
                                <h4><?php echo htmlspecialchars($match['other_username']); ?></h4>
                                <p><?php echo htmlspecialchars($match['last_message_preview'] ?? 'Henüz mesaj yok'); ?></p>
                            </div>
                            <span class="time"><?php echo htmlspecialchars($match['last_message_time_formatted']); ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php else: ?>
                <p class="no-matches">Henüz hiç eşleşmen yok veya eşleşmelerinle mesajlaşmadın.</p>
            <?php endif; ?>
        </div>
    </div>

</body>
</html>
