<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
require_once __DIR__ . '/../includes/auth_check.php'; // Corrected path

// Include database configuration
require_once __DIR__ . '/../config/db.php'; // Corrected path

$profile = [
    'username' => '',
    'birthdate' => null,
    'location' => '',
    'interests' => '',
    'bio' => '',
];
$photos = [];
$max_photos = 9;

// Establish database connection
$mysqli = connect_db();
$user_id = $_SESSION['user_id'];

// Fetch profile data
$sql_profile = "SELECT username, birthdate, location, interests, bio FROM profiles WHERE user_id = ?";
if ($stmt_profile = $mysqli->prepare($sql_profile)) {
    $stmt_profile->bind_param("i", $user_id);
    $stmt_profile->execute();
    $result_profile = $stmt_profile->get_result();
    if ($result_profile->num_rows === 1) {
        $profile = $result_profile->fetch_assoc();
    } else {
        // Create a default profile entry if none exists
        $sql_create_profile = "INSERT INTO profiles (user_id, username) VALUES (?, ?)";
        if ($stmt_create = $mysqli->prepare($sql_create_profile)) {
            $default_username = 'User' . $user_id;
            $stmt_create->bind_param("is", $user_id, $default_username);
            if ($stmt_create->execute()) {
                 $profile['username'] = $default_username; // Update profile array
            } else {
                error_log("Failed to create profile for user ID: " . $user_id . " Error: " . $stmt_create->error);
            }
            $stmt_create->close();
        } else {
             error_log("Failed to prepare profile creation statement: " . $mysqli->error);
        }
    }
    $stmt_profile->close();
} else {
    error_log("Failed to prepare profile fetch statement: " . $mysqli->error);
}

// Fetch photos
$sql_photos = "SELECT photo_id, file_path, upload_order FROM photos WHERE user_id = ? ORDER BY upload_order ASC";
if ($stmt_photos = $mysqli->prepare($sql_photos)) {
    $stmt_photos->bind_param("i", $user_id);
    $stmt_photos->execute();
    $result_photos = $stmt_photos->get_result();
    while ($row = $result_photos->fetch_assoc()) {
        // Ensure file_path is web-accessible relative to the public directory
        // Assuming file_path is stored like 'uploads/filename.jpg'
        $row['web_path'] = '../' . $row['file_path']; // Adjust path relative to public folder
        $photos[] = $row;
    }
    $stmt_photos->close();
} else {
     error_log("Failed to prepare photos fetch statement: " . $mysqli->error);
}

close_db($mysqli); // Close connection

$can_upload_more = count($photos) < $max_photos;

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profilimi Düzenle - Infernova</title>
    <link rel="stylesheet" href="../css/style.css"> <!-- Assuming a common CSS file -->
    <style>
        /* Add specific styles for profile page if needed */
        .photo-gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(100px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .photo-item {
            position: relative;
            border: 1px solid #ddd;
            border-radius: 4px;
            overflow: hidden;
            aspect-ratio: 1 / 1; /* Square photos */
        }
        .photo-item img {
            display: block;
            width: 100%;
            height: 100%;
            object-fit: cover; /* Crop images to fit */
        }
        .photo-item .delete-btn {
            position: absolute;
            top: 5px;
            right: 5px;
            background-color: rgba(255, 0, 0, 0.7);
            color: white;
            border: none;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            font-size: 12px;
            line-height: 20px;
            text-align: center;
            cursor: pointer;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        .photo-item:hover .delete-btn {
            opacity: 1;
        }
        .photo-upload {
            margin-top: 20px;
            padding: 15px;
            border: 1px dashed #ccc;
            border-radius: 4px;
            background-color: #f9f9f9;
        }
        input[type="file"] {
            border: none;
            padding: 0;
        }
        small {
            display: block;
            margin-top: 5px;
            color: #777;
        }
        .message {
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
            text-align: center;
            font-weight: bold;
        }
        .error-message {
            background-color: #ffebee; /* Light pink */
            color: #c62828; /* Dark red */
            border: 1px solid #ef9a9a;
        }
        .success-message {
            background-color: #e8f5e9; /* Light green */
            color: #2e7d32; /* Dark green */
            border: 1px solid #a5d6a7;
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="logo">
            <a href="dashboard.php">Infernova</a>
        </div>
        <div class="nav-links">
            <a href="profile.php" class="active">Profilim</a>
            <a href="find_matches.php">Eşleşme Bul</a>
            <a href="messages.php">Mesajlar</a>
            <a href="../includes/logout.php">Çıkış Yap</a>
        </div>
    </nav>

    <div class="container">
        <h2>Profilini Düzenle</h2>

        <?php
        // Display messages (errors or success)
        if (isset($_GET["status"])) {
            $message = "";
            $message_type = "success";
            switch ($_GET["status"]) {
                case "updated":
                    $message = "Profil başarıyla güncellendi.";
                    break;
                case "upload_error":
                    $message = "Fotoğraf yüklenirken bir hata oluştu.";
                    $message_type = "error";
                    break;
                case "invalid_file":
                    $message = "Geçersiz dosya türü veya boyutu.";
                    $message_type = "error";
                    break;
                case "max_photos":
                    $message = "Maksimum fotoğraf sayısına ulaştınız.";
                    $message_type = "error";
                    break;
                 case "db_error":
                    $message = "Veritabanı hatası oluştu.";
                    $message_type = "error";
                    break;
                case "delete_failed":
                    $message = "Fotoğraf silinirken bir hata oluştu.";
                    $message_type = "error";
                    break;
                case "photo_deleted":
                    $message = "Fotoğraf başarıyla silindi.";
                    break;
                // Add more cases as needed
            }
            echo "<div class=\"message " . $message_type . "-message\">" . htmlspecialchars($message) . "</div>";
        }
        ?>

        <form action="../includes/profile_process_fix.php" method="post" enctype="multipart/form-data">
            <h3>Profil Bilgileri</h3>
            <div class="form-group">
                <label for="username">Kullanıcı Adı:</label>
                <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($profile["username"] ?? "); ?>" required>
            </div>
            <div class="form-group">
                <label for="birthdate">Doğum Tarihi:</label>
                <input type="date" id="birthdate" name="birthdate" value="<?php echo htmlspecialchars($profile["birthdate"] ?? "); ?>">
            </div>
            <div class="form-group">
                <label for="location">Yaşadığı Yer:</label>
                <input type="text" id="location" name="location" value="<?php echo htmlspecialchars($profile["location"] ?? "); ?>">
            </div>
            <div class="form-group">
                <label for="interests">İlgi Alanları (virgülle ayırın):</label>
                <input type="text" id="interests" name="interests" value="<?php echo htmlspecialchars($profile["interests"] ?? "); ?>">
            </div>
            <div class="form-group">
                <label for="bio">Hakkında:</label>
                <textarea id="bio" name="bio" rows="4"><?php echo htmlspecialchars($profile["bio"] ?? "); ?></textarea>
            </div>

            <h3>Fotoğraflar (Maksimum <?php echo $max_photos; ?>)</h3>
            <div class="photo-gallery">
                <?php foreach ($photos as $photo): ?>
                    <div class="photo-item">
                        <img src="<?php echo htmlspecialchars($photo["web_path"] ?? "); ?>" alt="Fotoğraf <?php echo $photo["photo_id"]; ?>">
                        <!-- Basic delete form (improve with JS later) -->
                        <form action="../includes/profile_process_fix.php" method="post" style="position: absolute; top: 5px; right: 5px;">
                            <input type="hidden" name="action" value="delete_photo">
                            <input type="hidden" name="photo_id" value="<?php echo $photo["photo_id"]; ?>">
                            <button type="submit" class="delete-btn" title="Fotoğrafı Sil" onclick="return confirm("Bu fotoğrafı silmek istediğinizden emin misiniz?");">×</button>
                        </form>
                    </div>
                <?php endforeach; ?>
                <?php if (empty($photos)): ?>
                    <p>Henüz fotoğraf yüklenmemiş.</p>
                <?php endif; ?>
            </div>

            <?php if ($can_upload_more): ?>
            <div class="photo-upload form-group">
                <label for="photos">Yeni Fotoğraf Yükle:</label>
                <input type="file" id="photos" name="photos[]" accept="image/jpeg, image/png, image/webp" multiple>
                <small>Kalan Yükleme Hakkı: <?php echo $max_photos - count($photos); ?>. Maksimum boyut: 2MB.</small>
            </div>
            <?php else: ?>
            <p>Maksimum fotoğraf sayısına ulaştınız (<?php echo $max_photos; ?>).</p>
            <?php endif; ?>

            <button type="submit" name="action" value="update_profile">Profili Güncelle</button>
        </form>

    </div>

</body>
</html>