<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
require_once __DIR__ . '/../includes/auth_check.php'; // Corrected path

$user_id = $_SESSION['user_id'];
$match_id = filter_input(INPUT_GET, 'match_id', FILTER_VALIDATE_INT);
$other_user_id = filter_input(INPUT_GET, 'user_id', FILTER_VALIDATE_INT);

if (!$match_id || !$other_user_id) {
    // Redirect back or show an error message
    header("Location: messages.php?error=invalid_chat");
    exit;
}

// Include database configuration
require_once __DIR__ . '/../config/db.php'; // Corrected path
$mysqli = connect_db(); // Establish connection

// Fetch other user's username
$other_username = "Bilinmeyen Kullanıcı"; // Default username
$sql_user = "SELECT username FROM profiles WHERE user_id = ?";
if ($stmt_user = $mysqli->prepare($sql_user)) {
    $stmt_user->bind_param("i", $other_user_id);
    $stmt_user->execute();
    $result_user = $stmt_user->get_result();
    if ($row_user = $result_user->fetch_assoc()) {
        $other_username = $row_user['username'];
    }
    $stmt_user->close();
} else {
    error_log("Conversation DB Error (fetch username): " . $mysqli->error);
}

// Fetch messages for this match
$messages = [];
$sql_messages = "SELECT message_id, sender_id, content, created_at FROM messages WHERE match_id = ? ORDER BY created_at ASC";
if ($stmt_messages = $mysqli->prepare($sql_messages)) {
    $stmt_messages->bind_param("i", $match_id);
    $stmt_messages->execute();
    $result_messages = $stmt_messages->get_result();
    while ($row = $result_messages->fetch_assoc()) {
        // Format timestamp
        $dt = new DateTime($row['created_at']);
        $row['time_formatted'] = $dt->format('H:i'); // Example format, adjust as needed
        $messages[] = $row;
    }
    $stmt_messages->close();
} else {
    error_log("Conversation DB Error (fetch messages): " . $mysqli->error);
}

// Mark messages sent by the other user as read
$sql_mark_read = "UPDATE messages SET is_read = TRUE WHERE match_id = ? AND receiver_id = ? AND sender_id = ? AND is_read = FALSE";
if ($stmt_read = $mysqli->prepare($sql_mark_read)) {
    $stmt_read->bind_param("iii", $match_id, $user_id, $other_user_id);
    $stmt_read->execute();
    $stmt_read->close();
} else {
    error_log("Conversation DB Error (mark read): " . $mysqli->error);
}

close_db($mysqli);

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($other_username); ?> ile Konuşma</title>
    <link rel="stylesheet" href="../css/style.css"> <!-- Assuming a common CSS file -->
    <style>
        body {
            display: flex;
            flex-direction: column;
            height: 100vh;
            margin: 0;
            font-family: sans-serif;
            background-color: #f4f7f6;
        }
        header {
            background-color: #e91e63; /* Pink theme */
            color: white;
            padding: 10px 20px;
            display: flex;
            align-items: center;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        header a {
            color: white;
            text-decoration: none;
            font-size: 1.2em;
            margin-right: 15px;
        }
        header h2 {
            margin: 0;
            font-size: 1.2em;
            font-weight: 500;
        }
        .message-list {
            flex-grow: 1;
            overflow-y: auto;
            padding: 20px;
            display: flex;
            flex-direction: column;
        }
        .message {
            margin-bottom: 15px;
            display: flex;
            max-width: 75%;
            align-self: flex-start; /* Default to received */
        }
        .message .content {
            padding: 10px 15px;
            border-radius: 18px;
            word-wrap: break-word;
            background-color: #fff; /* Received message background */
            border: 1px solid #eee;
            box-shadow: 0 1px 2px rgba(0,0,0,0.05);
        }
        .message.sent {
            align-self: flex-end; /* Sent messages align to right */
        }
        .message.sent .content {
            background-color: #dcf8c6; /* Sent message background */
            border: none;
        }
        .message .time {
            font-size: 0.75em;
            color: #999;
            margin-top: 5px;
            text-align: right;
        }
        .message.received .time {
            text-align: left;
        }
        .message-form {
            padding: 15px;
            border-top: 1px solid #ccc;
            background-color: #f9f9f9;
            display: flex;
            align-items: center;
        }
        .message-form input[type="text"] {
            flex-grow: 1;
            padding: 12px 15px;
            border: 1px solid #ccc;
            border-radius: 20px;
            margin-right: 10px;
            font-size: 1em;
        }
        .message-form button {
            padding: 12px 20px;
            border: none;
            background-color: #e91e63; /* Pink theme */
            color: white;
            border-radius: 20px;
            cursor: pointer;
            font-size: 1em;
            transition: background-color 0.2s ease;
        }
        .message-form button:hover {
            background-color: #c2185b;
        }
        .no-messages {
            text-align: center;
            color: #888;
            margin-top: 50px;
        }
        /* Basic emoji support styling (optional) */
        .emoji { font-size: 1.2em; }
    </style>
</head>
<body>
    <header>
        <a href="messages.php">&larr; Geri</a>
        <h2><?php echo htmlspecialchars($other_username); ?></h2>
    </header>

    <div class="message-list" id="message-list">
        <?php foreach ($messages as $msg): ?>
            <div class="message <?php echo ($msg["sender_id"] == $user_id) ? 'sent' : 'received'; ?>">
                <div class="content">
                    <?php echo nl2br(htmlspecialchars($msg["content"])); ?>
                    <div class="time"><?php echo htmlspecialchars($msg["time_formatted"]); ?></div>
                </div>
            </div>
        <?php endforeach; ?>
        <?php if (empty($messages)): ?>
            <p class="no-messages">Mesajlaşmaya başlayın!</p>
        <?php endif; ?>
    </div>

    <form action="../includes/send_message_process_fix.php" method="post" class="message-form" id="message-form">
        <input type="hidden" name="match_id" value="<?php echo $match_id; ?>">
        <input type="hidden" name="receiver_id" value="<?php echo $other_user_id; ?>">
        <input type="text" name="content" placeholder="Mesajınızı yazın..." autocomplete="off" required>
        <!-- Add emoji picker button later -->
        <button type="submit">Gönder</button>
    </form>

    <script>
        // Auto-scroll to bottom on page load
        const messageList = document.getElementById("message-list");
        messageList.scrollTop = messageList.scrollHeight;

        // Optional: Basic AJAX for sending messages without full page reload (more advanced)
        /*
        const messageForm = document.getElementById("message-form");
        messageForm.addEventListener("submit", function(event) {
            event.preventDefault();
            const formData = new FormData(messageForm);
            const messageInput = messageForm.querySelector("input[name=\"content\"]");
            const messageText = messageInput.value.trim();

            if (messageText === "") return;

            // Optimistically add the message to the UI
            const now = new Date();
            const timeFormatted = now.getHours().toString().padStart(2, '0') + ':' + now.getMinutes().toString().padStart(2, '0');
            const newMessageDiv = document.createElement('div');
            newMessageDiv.classList.add('message', 'sent');
            newMessageDiv.innerHTML = `
                <div class="content">
                    ${messageText.replace(/</g, "&lt;").replace(/>/g, "&gt;")}
                    <div class="time">${timeFormatted}</div>
                </div>
            `;
            messageList.appendChild(newMessageDiv);
            messageList.scrollTop = messageList.scrollHeight;
            messageInput.value = ''; // Clear input

            // Send the message via fetch
            fetch('../includes/send_message_process_fix.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json()) // Assuming your PHP script returns JSON
            .then(data => {
                if (!data.success) {
                    console.error('Mesaj gönderilemedi:', data.error);
                    // Optionally remove the optimistically added message or mark it as failed
                    newMessageDiv.style.opacity = '0.5'; // Indicate failure
                }
            })
            .catch(error => {
                console.error('Mesaj gönderme hatası:', error);
                newMessageDiv.style.opacity = '0.5'; // Indicate failure
            });
        });
        */
    </script>

</body>
</html>